'''
Created on 10.11.2017
    
	System class providing a interface to a number of cooling systems.  
    Copyright (C) 2018  Miika Rama

    This program is free software: you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation, either version 3 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program.  If not, see <https://www.gnu.org/licenses/>.
	
@author: Miika Rama
'''
from Classes.Resource import Resource
from Classes.EnergySupply import EnergySupply
from Classes.Producer import Producer
from Classes.Consumer import Consumer
from Classes.CoolingSystem import CoolingSystem
from Classes.Network import Network

import copy

class System(object):
    '''
    classdocs
    '''

    def __init__(self):
        self.systems = []
        self.resources = []
        #self.supply_techs = {}
        #self.cooling_techs = {}
        self.interest_rate = 5
        self.costs = []
        self.profitability = []
        self.location = ""
        
    def initialise(self, location):
        
        # initialising resources based on location, e.g. "Spain"
        try:
            # resources prices (res) and projected future prices (pro)
            res = open("Data/resources_"+location+".txt", 'r')
            pro = open("Data/resources_projection_"+location+".txt", 'r')
            
            # cleaning up possible previous definitions
            self.resources = []
            
            # reading the file
            data1 = res.readlines()
            data2 = pro.readlines()
            
            # disecting items from the data (header line)
            items = data1[0].replace('\n', '').split('\t')

            # defining resources based on data
            for i in range(0,len(items)):
                ts = []
                p = []
                for j in range(1,len(data1)):
                    line = data1[j].replace('\n', '').split('\t')
                    ts.append(float(line[i]))
                for k in range(1,len(data2)):
                    line = data2[k].replace('\n', '').split('\t')
                    p.append(float(line[i]))
                self.resources.append(Resource(items[i], ts, p))
        except:
            print('Error reading resources file!!')
        finally:
            res.close()
            pro.close()
    
        # defining the location for the cooling systems to be defined
        self.location = location
    
    def setCoolingPrices(self, allSystems, systemName, energyTariff, capacityCharge):
        # setting prices (energy tariffs �/MWh, capacity charge �/kW) for cooling 
        if allSystems:
            for i in range(0, len(self.systems)):
                self.systems[i].energyTariff = energyTariff
                self.systems[i].capacityCharge = capacityCharge
        else:
            for i in range(0,len(self.systems)):
                if self.systems[i].name == systemName:
                    self.systems[i].energyTariff = energyTariff
                    self.systems[i].capacityCharge = capacityCharge
                           
                    
    def initNetwork(self, systemName,scenName):
        path_name = '.\\GIS\\'+systemName+'\\'
        s_data = path_name+'system_data_'+scenName+'.txt'
        d_netw = path_name+'dh_network_'+scenName+'.txt'
        d_prod = path_name+'prod_con_'+scenName+'.txt'
        d_node = path_name+'dh_nodes_'+scenName+'.txt'
        for i in range(0,len(self.systems)):
            if self.systems[i].name == systemName:
                #self.systems[i].network.initNetwork()
                self.systems[i].network.readNetworkDefinition(s_data, d_netw, d_prod, d_node)
                return
        print('No system found with a specified name! No storage input returned.')
    
    def setResourceStatus(self, resource, status):
        # disabling specific resources, e.g. excess heat as a resource for absorption cooling:
        # sets solar resource very low (value/10^6) or the price very high (value*1000)
        found = False
        for i in range(0, len(self.resources)):
            if self.resources[i].name == resource:
                found = True
                break
        if found == True:
            if status == False:
                self.resources[i].disable()
            else:
                self.resources[i].enable()
        else:
            print("Resource not found, nothing disabled or enabled.")
            
    def setYear(self, year):
        # default year is 1, if a different year is specified costs are multiplied 
        # by a corresponding value from resource_projection file.
        for i in range(0,len(self.systems)):
            self.systems[i].year = year

    def addCoolingSystem(self, name, hasNetwork, tariff, charge):
        self.systems.append(CoolingSystem(name, self, tariff, charge))
        if hasNetwork:
            self.systems[len(self.systems)-1].network = Network()
            
    def addConsumer(self, systemName, id_number, demand):
        for i in range(0,len(self.systems)):
            if self.systems[i].name == systemName:
                self.systems[i].consumers.append(Consumer(id_number, demand))
                return
        print('No system found with a specified name! Consumer not added.')
    
    def addProducer(self, systemName, id_number, cap, ne, ee, he, inv, om):
        for i in range(0,len(self.systems)):
            if self.systems[i].name == systemName:
                self.systems[i].producers.append(Producer(id_number, cap, ne, ee, he, inv, om, 20))
                return
        print('No system found with a specified name! Producer not added.')

    def addSimpleProducer(self, systemName, tech, id_number, cap):
        
        # a simple way to add a producer:
        # only id and capacity given (id, cap), default values from a text file used for other parameters
        # valid values for tech: "Compressior chiller", "Absorption chiller", "Free cooling"
        
        try:
            f = open("Data/cooling_techs.txt", 'rt')
            
            # reading the file
            lines = f.readlines()
            
            # disecting data 
            data = []
            for i in range(0, len(lines)):
                data.append(lines[i].replace('\n', '').split('\t'))

        except:
            print('Error in reading cooling technologies (cooling_techs.txt) data!')
        finally:
            f.close()        
        
        for k in range(0,len(self.systems)):
            if self.systems[k].name == systemName:
                for i in range(1,len(data[0])):
                    if data[0][i] == tech:
                        self.systems[k].producers.append(Producer(id_number, 
                                                                  cap, float(data[2][i]), float(data[3][i]), 
                                                                  float(data[4][i]), float(data[5][i]), float(data[6][i]),
                                                                  float(data[7][i])))
                        return

                print('Specified technology not found! Producer not added.')
                return
        print('No system found with a specified name! Producer not added.')
        
    def addSupply(self, systemName, id_number, tech, resource, ee, he, cap, inv, om):
        for i in range(0,len(self.systems)):
            if self.systems[i].name == systemName:
                self.systems[i].supply.append(EnergySupply(id_number, tech, resource, ee, he, cap, inv, om, 20))
                return
        print('No system found with a specified name! Energy supply not added.')

    def addSimpleSupply(self, systemName, tech, fuel, id_number, cap):
        
        # a simple way to add an energy supply unit:
        # only id and capacity given (id) and (cap) 
        # default values from a text file for other parameters based on specified (tech) and (fuel)
        # tech: "CHP", "Boiler", "Collectors", "PVs", "Grid connection", "Excess heat", "Cooling source"
        # fuel: "LFO", "HFO", "NG", "Chips", "Pellets", "Sun"...       
               
        try:
            f = open("Data/supply_techs.txt", 'rt')
            
            # reading the file
            lines = f.readlines()
            
            # accessing data
            data = []
            for i in range(0, len(lines)):
                data.append(lines[i].replace('\n', '').split('\t'))

        except:
            print('Error in reading supply technologies (supply_techs.txt) data!')
        finally:
            f.close()        
            
        for k in range(0,len(self.systems)):
            if self.systems[k].name == systemName:
                for i in range(1,len(data[0])):
                    self.systems[k].supply.append(EnergySupply(id_number, data[0][i], fuel, 
                                                               float(data[1][i]), float(data[2][i]), cap, 
                                                               float(data[4][i]), float(data[5][i]), float(data[6][i])))
                    return

                print('Specified technology not found! Producer not added.')
                return
            
        print('No system found with a specified name! Producer not added.')
  
    def defineStorage(self, systemName, storageType, props):
        # valid storageType values: "dc storage", "elec storage", "heat storage", "cold storage"
        for i in range(0,len(self.systems)):
            if self.systems[i].name == systemName:
                if storageType in self.systems[i].storages:
                    self.systems[i].storages[storageType].capacity = props[0]
                    self.systems[i].storages[storageType].losses = props[1]
                    self.systems[i].storages[storageType].initCharge = props[2]
                    self.systems[i].storages[storageType].loadingCost = props[3]
                    self.systems[i].storages[storageType].unloadingCost = props[4]
                    self.systems[i].storages[storageType].loadingRatio = props[5]
                    self.systems[i].storages[storageType].unloadingRatio = props[6]
                    self.systems[i].storages[storageType].loadingEff = props[7]
                    self.systems[i].storages[storageType].unloadingEff = props[8]                                        
                    if props[9] == 0:
                        # calculate investment
                        self.systems[i].storages[storageType].spec_inv = 0                                                
                    else:
                        self.systems[i].storages[storageType].spec_inv = props[9] 
                    
                else:
                    print('Unknown storage type!')
                return    
        print('No system found with a specified name! Consumer not added.')

    def defineSimpleStorage(self, systemName, storageType, cap, loss, load, unload, inv):
        # valid storageType values: "dc storage", "elec storage", "heat storage", "cold storage"
        for i in range(0,len(self.systems)):
            if self.systems[i].name == systemName:
                if storageType in self.systems[i].storages:
                    self.systems[i].storages[storageType].capacity = cap
                    self.systems[i].storages[storageType].losses = loss
                    self.systems[i].storages[storageType].initCharge = 0
                    self.systems[i].storages[storageType].loadingCost = 0.00001
                    self.systems[i].storages[storageType].unloadingCost = 0.00001
                    self.systems[i].storages[storageType].loadingRatio = min(load/cap, 1)
                    self.systems[i].storages[storageType].unloadingRatio = min(unload/cap, 1)
                    self.systems[i].storages[storageType].loadingEff = 1
                    self.systems[i].storages[storageType].unloadingEff = 1                                        
                    if inv == 0:
                        # calculate investment
                        self.systems[i].storages[storageType].spec_inv = 0                                                
                    else:
                        self.systems[i].storages[storageType].spec_inv = inv
                else:
                    print('Unknown storage type!')
                return    
        print('No system found with a specified name! Consumer not added.')
    
    def runOptimisation(self, allSystems, systemName, period):
        
        # if all is True, then run all the systems - if not, only the system with the specified name
        if allSystems:
            for i in range(0,len(self.systems)):
                if self.systems[i].buildAndRun(period) == 0:
                    return 0
            return 1
        else: 
            for i in range(0,len(self.systems)):
                if self.systems[i].name == systemName:
                    return self.systems[i].buildAndRun(period) 

    def runOptimisationForMultipleYears(self, systemName, years):
        # returning a dict with "systemName" as the key
        results = {}
        
        # if not systemName is defined, running optimisation for all specified systems
        if systemName == "":
            for i in range(0,len(self.systems)):
                yearlyRuns = []
                for j in range(0, years):
                    self.setYear(j+1)
                    if self.systems[i].buildAndRun(8760) == 0:
                        return 0
                    #results[self.systems[i].name+" ("+str(j+1)+")"] = self.systems[i]
                    yearlyRuns.append(copy.deepcopy(self.systems[i]))
                results[self.systems[i].name] = yearlyRuns
                    
        # running optimisation only for a system with the specified name
        else: 
            for i in range(0,len(self.systems)):
                if self.systems[i].name == systemName:
                    yearlyRuns = []
                    for j in range(0, years):
                        self.setYear(j+1)
                        if self.systems[i].buildAndRun(8760) == 0:
                            return 0
                        yearlyRuns.append(copy.deepcopy(self.systems[i]))
                    results[self.systems[i].name] = yearlyRuns

        return results    
                                                             
    def getUsedResources(self, systemName):
        for i in range(0,len(self.systems)):
            if self.systems[i].name == systemName:
                return self.systems[i].usedResources       
            
    def getSupplyInput(self, systemName, id_number):
        # returned value corresponds to the resource defined for the unit
        for i in range(0,len(self.systems)):
            if self.systems[i].name == systemName:
                for j in range(0, len(self.systems[i].supply)):
                    if self.systems[i].supply[j].node_id == id_number:
                        data = []
                        for k in range(0, len(self.systems[i].supply[j].consumption)):
                            data.append(self.systems[i].supply[j].consumption[k])
                        return data
        print('No system found with a specified name! No supply input returned.')
        
    def getSupplyOutput(self, systemName, id_number, commodity):
        # commodity can be "Electricity" or "Heat"
        for i in range(0,len(self.systems)):
            if self.systems[i].name == systemName:
                for j in range(0, len(self.systems[i].supply)):
                    if self.systems[i].supply[j].node_id == id_number:
                        if commodity == 'Electricity':
                            data = []
                            for k in range(0, len(self.systems[i].supply[j].elec_production)):
                                data.append(self.systems[i].supply[j].elec_production[k])
                            return data
                            #return self.systems[i].supply[j].elec_production
                        elif commodity == 'Heat':
                            data = []
                            for k in range(0, len(self.systems[i].supply[j].heat_production)):
                                data.append(self.systems[i].supply[j].heat_production[k])
                            return data
                            #return self.systems[i].supply[j].heat_production
                        else:
                            print('Commodity not found!')
        print('No system found with a specified name! No supply input returned.')
    
    def getProducerInput(self, systemName, id_number, commodity):
        # commodity can be "Electricity", "Heat" or "Natural"
        for i in range(0,len(self.systems)):
            if self.systems[i].name == systemName:
                for j in range(0, len(self.systems[i].producers)):
                    if self.systems[i].producers[j].node_id == id_number:
                        if commodity == 'Electricity':
                            data = []
                            for k in range(0, len(self.systems[i].producers[j].elec_cons)):
                                data.append(self.systems[i].producers[j].elec_cons[k])
                            return data
                            #return self.systems[i].producers[j].elec_cons
                        elif commodity == 'Heat':
                            data = []
                            for k in range(0, len(self.systems[i].producers[j].heat_cons)):
                                data.append(self.systems[i].producers[j].heat_cons[k])
                            return data
                            #return self.systems[i].producers[j].heat_cons
                        elif commodity == 'Natural':
                            data = []
                            for k in range(0, len(self.systems[i].producers[j].natr_cons)):
                                data.append(self.systems[i].producers[j].natr_cons[k])
                            return data
                            #return self.systems[i].producers[j].natr_cons
                        else:
                            print('Commodity not found!')
        print('No system found with a specified name! No producer input returned.')
    
    def getProducerOutput(self, systemName, id_number):
        # output always district cooling
        for i in range(0,len(self.systems)):
            if self.systems[i].name == systemName:
                for j in range(0, len(self.systems[i].producers)):
                    if self.systems[i].producers[j].node_id == id_number:
                        if len(self.systems[i].producers[j].production) > 0:
                            data = []
                            for k in range(0, len(self.systems[i].producers[j].production)):
                                data.append(self.systems[i].producers[j].production[k])
                            return data
                            #return self.systems[i].producers[j].production
                        else:
                            print('No output found!')
        print('No system found with a specified name or id number! No producer output returned.')
    
    def getStorageInput(self, systemName, storageType):
        # storageType can be "dc storage", "elec storage", "heat storage" or "cold storage"
        for i in range(0,len(self.systems)):
            if self.systems[i].name == systemName:
                load = []
                for k in range(0, len(self.systems[i].storages[storageType].load)):
                    load.append(self.systems[i].storages[storageType].load[k])
                #load = self.systems[i].storages[storageType].load
                if load == []:
                    print('Storage type not recognized!')
                else:
                    return load
        print('No system found with a specified name! No storage input returned.')
    
    def getStorageOutput(self, systemName, storageType):
        # storageType can be "dc storage", "elec storage", "heat storage" or "cold storage"
        for i in range(0,len(self.systems)):
            if self.systems[i].name == systemName:
                unload = []
                for k in range(0, len(self.systems[i].storages[storageType].unload)):
                    unload.append(self.systems[i].storages[storageType].unload[k])
                #unload = self.systems[i].storages[storageType].unload
                if unload == []:
                    print('Storage type not recognized!')
                else:
                    return unload
        print('No system found with a specified name! No storage input returned.')
                    
    def getStorageState(self, systemName, storageType, relative):
        # storageType can be "dc storage", "elec storage", "heat storage" or "cold storage"
        for i in range(0,len(self.systems)):
            if self.systems[i].name == systemName:
                unload = self.systems[i].storages[storageType].unload[:]
                if len(unload) == 0:
                    print('Storage type not recognized!')
                    return 0
                load = self.systems[i].storages[storageType].load[:]
                period = len(self.systems[i].storages[storageType].unload)
                state = [0]*period
                
                # state of the storage in kWh
                state[0] = max(0, state[0]*(1-self.systems[i].storages[storageType].losses) + load[0] - unload[0])
                for k in range(1, period):
                    state[k] = max(0, state[k-1]*(1-self.systems[i].storages[storageType].losses) + load[k] - unload[k])
                
                # calculating relative charge (if necessary)
                if relative and self.systems[i].storages[storageType].capacity > 0:
                    for k in range(0, period):
                        state[k] = state[k] / self.systems[i].storages[storageType].capacity
                
                return state
            
        print('No system found with a specified name! No storage state returned.')
        
    def getTotalDemand(self, systemName):
        for i in range(0,len(self.systems)):
            if self.systems[i].name == systemName:
                return self.systems[i].getTotalDemand()

    def getCoolingDemand(self, systemName):
        for i in range(0,len(self.systems)):
            if self.systems[i].name == systemName:
                return self.systems[i].getCoolingDemand()
            
    def getUtilisationRate(self, systemName, isSupply, id_number):
        for i in range(0,len(self.systems)):
            if self.systems[i].name == systemName:
                if (isSupply):
                    for j in range(0, len(self.systems[i].supply)):                    
                        if self.systems[i].supply[j].node_id == id_number:
                            return self.systems[i].supply[j].utilisation
                    print("Energy supply id number not recognized! Zero utilisation rate returned.")
                    return 0
                else:
                    for j in range(0, len(self.systems[i].producers)):                    
                        if self.systems[i].producers[j].node_id == id_number:
                            return self.systems[i].producers[j].utilisation
                    print("Cooling producer id number not recognized! Zero utilisation rate returned.")
                    return 0                                    
        print('No system found with a specified name! Zero utilisation rate returned.')
        return 0            
    
    def getNumberOfCycles(self, systemName, storageType):
        # storageType can be "dc storage", "elec storage", "heat storage" or "cold storage"
        for i in range(0,len(self.systems)):
            if self.systems[i].name == systemName:
                cycles = self.systems[i].storages[storageType].cycles
                if cycles == 0:
                    print('Storage type not recognized or storage not used! Zero cycles returned.')
                    return 0
                return cycles
        print('No system found with a specified name! Returning zero cycles.')
        return 0
     
    def runEconomicAnalysis(self, years):
        
        # calculating cumulative costs (only costs) and profitability (costs + revenues), k� as unit, for a defined number of years
        
        costs = [0] * years
        profitability = [0] * years
        
        supplyInvestment = []
        coolingInvestment = []
        networkInvestment = []
        storageInvestment = []
        
        supplyOM = 0
        coolingOM = 0
        networkOM = 0
        
        operationalCosts = 0
        operationalRevenues = 0
        
        # going through each system        
        for i in range(0,len(self.systems)):

            # investment and operation/maintenance costs for energy supply 
            for j in range(0,len(self.systems[i].supply)):
                supplyInvestment.append(self.systems[i].supply[j].calculateInvestment()/1000)
                supplyOM += self.systems[i].supply[j].calculateInvestment()/1000 * self.systems[i].supply[j].om

            # investment and operation/maintenance costs for cooling production 
            for k in range(0,len(self.systems[i].producers)):
                coolingInvestment.append(self.systems[i].producers[k].calculateInvestment()/1000)
                coolingOM += self.systems[i].producers[k].calculateInvestment()/1000 * self.systems[i].producers[k].om
            
            # investment and operation/maintenance costs for distribution network
            if self.systems[i].network != []:
                networkInvestment.append(self.systems[i].network.calculateInvestment()/1000)
                networkOM += self.systems[i].network.calculateInvestment()/1000 * self.systems[i].network.om
            else:
                networkInvestment.append(0.0)
                networkOM += 0.0          

            # investment for different types of storages
            for key in self.systems[i].storages:
                storageInvestment.append(self.systems[i].storages[key].calculateInvestment()/1000)
                #print(self.systems[i].storages[key].spec_inv)
                #print(self.systems[i].storages[key].capacity)

            #print("varastoinvestointi:")
            #print(sum(storageInvestment))

            # multiplying use of the resource and the price of the resource 
            for m in self.systems[i].usedResources:
                for o in range(0,len(self.resources)):
                    data = []
                    if self.resources[o].name == m and not m == "Sun":
                        data = self.resources[o].availability
                        for p in range(0,len(self.systems[i].usedResources[m])):
                            # EUR/MWh / 1000 => EUR / kWh * kWh = EUR
                            operationalCosts += (data[p]/1000)*self.systems[i].usedResources[m][p]
            
            # EUR to kEUR
            operationalCosts = operationalCosts / 1000

            totalDemand = self.systems[i].getTotalDemand()
                      
            # calculating revenues (EUR) from energy tariffs, assuming a time series for energy tariff (EUR/MWh)
            energyRevenues = 0    
            for l in range(0, len(totalDemand)):                    
                energyRevenues += (totalDemand[l]/1000) * self.systems[i].energyTariff[l]
            
            # calculating revenues from capacity charges (EUR/kW)
            chargeRevenues = 0    
            for l in range(0, len(self.systems[i].consumers)):
                chargeRevenues += max(self.systems[i].consumers[l].demand) * self.systems[i].capacityCharge

            # sum and EUR to kEUR        
            operationalRevenues = ( energyRevenues + chargeRevenues ) / 1000
                
            costs[0] = sum(supplyInvestment) + sum(coolingInvestment) + sum(networkInvestment) + sum(storageInvestment)
            profitability[0] = -costs[0]
    
            thefile = open("Costs and profits ("+self.systems[i].name+").txt", 'w')                
            thefile.write("%s;%s;\n" % ('Cumulative costs', 'Profitability'))
            thefile.write("%r;%r;\n" % (costs[0], profitability[0]))
            
            self.costs = []
            self.profitability = []
            
            for i in range(1,years):
                
                df = 1 / (1+(self.interest_rate/100))**i
                
                yearlyCosts = df * (supplyOM + coolingOM + networkOM + operationalCosts)
                yearlyRevenues = df * (operationalRevenues)
                                
                costs[i] = costs[i-1] + (yearlyCosts)
                profitability[i] = profitability[i-1] + (yearlyRevenues - yearlyCosts)
                self.costs.append(costs[i])
                self.profitability.append(profitability[i])
                
                thefile.write("%r;%r;\n" % (costs[i], profitability[i]))
                
            thefile.close()    

    def getCumulativeCosts(self):
        # returns time series of cumulative costs
        if len(self.costs) < 1:
            print("No costs calculated...")
            return [0.0]
        else:
            return self.costs
    
    def getCumulativeCostsAndRevenues(self):
        # returns time series of cumulative costs and revenues (=profitability)
        if len(self.profitability) < 1:
            print("No costs calculated...")
            return [0.0]
        else:
            return self.profitability

    def getPaybackPeriod(self):
        
        # calculating payback period (if any) in number of years 
        if len(self.profitability) < 1:
            print("No costs calculated...")
            return [0.0]
        else:
            found = False
            for i in range(0, len(self.profitability)):
                if self.profitability[i] > 0:
                    found = True
                    break
            if found:
                partial_year = (0.0 - self.profitability[i-1] ) / (self.profitability[i] - self.profitability[i-1])
                return float(i+1) + partial_year 
            else:
                print("No payback period...")
                return float(len(self.profitability)+1)

    def runEconomicAnalysisForMultipleYears(self, optimisationRuns):
        return 0

    def getOperationalEmissions(self, systemName, location):
        
        # returning a dictionary with resource as key and (yearly) emissions as kgCO2
        
        results = {}
        emissions = {}

        try:
            f = open("Data/emissions_"+location+".txt", 'r')
            data = f.readlines()
            items = data[0].replace('\n', '').split('\t')
            for i in range(1,len(items)):
                line1 = data[1].replace('\n', '').split('\t')
                line2 = data[2].replace('\n', '').split('\t')
                line3 = data[3].replace('\n', '').split('\t')
                # total emissions = consumption + production + plant (CHP/boiler) as g/kWh
                emissions[items[i]] = float(line1[i]) + float(line2[i]) + float(line3[i]) 
        except:
            print('Error reading emissions file!!')
        finally:
            f.close()
        
        # if no systemName is specified, calculating emissions from all systems
        if systemName == "":
            for i in range(0, len(self.systems)):
                for k in self.systems[i].usedResources.keys():
                    if k in results:
                        results[k] += sum(self.systems[i].usedResources[k]) * emissions[k] / 1000
                    else:
                        results[k] = sum(self.systems[i].usedResources[k]) * emissions[k] / 1000
            return results

        # returning emissions from the specified system
        else:
            for i in range(0,len(self.systems)):
                if self.systems[i].name == systemName:
                    for k in self.systems[i].usedResources.keys():
                        results[k] = sum(self.systems[i].usedResources[k]) * emissions[k] / 1000
                    return results
    
    def getEmbeddedEmissions(self, systemName):
        
        results = {}

        # preformatting results dictionary
        results["Energy supply"] = 0
        results["Cooling production"] = 0
        results["Network"] = 0
        results["Storages"] = 0

        # if no systemName is specified, calculating emissions from all systems
        if systemName == "":
            for i in range(0, len(self.systems)):
                embeddedEmissions = self.systems[i].getEmbeddedEmissions()
                results["Energy supply"] += embeddedEmissions[1]
                results["Cooling production"] += embeddedEmissions[2]
                results["Network"] += embeddedEmissions[0]
                results["Storages"] += embeddedEmissions[3]
            return results
                
        # returning emissions from the specified system
        else:
            for i in range(0,len(self.systems)):
                if self.systems[i].name == systemName:
                    embeddedEmissions = self.systems[i].getEmbeddedEmissions()
                    results["Energy supply"] += embeddedEmissions[1]
                    results["Cooling production"] += embeddedEmissions[2]
                    results["Network"] += embeddedEmissions[0]
                    results["Storages"] += embeddedEmissions[3]
                    return results
        
        # returning dictionary with zero emissions, system not found
        print("System not found!") 
        return results
    
    def getResultsSummary(self, systemName, location):
        # compiles a predefined set of results for all systems or a specified system
        
        results = {}
        totalDemand = 0.0
        totalCosts = 0.0
        
        # emissions for operation and materials/constructions
        operationalEmissions = 0.0
        embeddedEmissions = 0.0
        
        if systemName == "":
            for i in range(0,len(self.systems)):
                embeddedEmissions += sum(self.systems[i].getEmbeddedEmissions())
                operationalEmissions += self.systems[i].getTotalEmissions(location)
                totalDemand += sum(self.systems[i].getTotalDemand())
                totalCosts += self.systems[i].totalCosts
                if totalCosts == 0:
                    print("Costs zero...?")
        else: 
            for i in range(0,len(self.systems)):
                if self.systems[i].name == systemName:
                    embeddedEmissions += sum(self.systems[i].getEmbeddedEmissions())
                    operationalEmissions += self.systems[i].getTotalEmissions(location)
                    totalDemand += sum(self.systems[i].getTotalDemand())
                    totalCosts += self.systems[i].totalCosts
                    if totalCosts == 0:
                        print("Costs zero...?")
                    break 
        
        totalEmissions = operationalEmissions + embeddedEmissions
        
        results['Total cooling (MWh)'] = totalDemand/1000
        results['Total costs (ke)'] = totalCosts/1000
        results['Total emissions (tCO2)'] = totalEmissions/1000
        results['Operation emissions (tCO2)'] = operationalEmissions/1000
        results['Embedded emissions (tCO2)'] = embeddedEmissions/1000
        results['Share of embedded emissions (%)'] = embeddedEmissions / totalEmissions * 100
        results['Cooling production costs (e/MWh)'] = totalCosts / (totalDemand/1000)
        results['Cooling emissions (kgCO2/MWh)'] = totalEmissions / (totalDemand/1000)
        return results
